<?php

/**
 * ---------------------------------------------------------------------
 *
 * GLPI - Gestionnaire Libre de Parc Informatique
 *
 * http://glpi-project.org
 *
 * @copyright 2015-2026 Teclib' and contributors.
 * @licence   https://www.gnu.org/licenses/gpl-3.0.html
 *
 * ---------------------------------------------------------------------
 *
 * LICENSE
 *
 * This file is part of GLPI.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 * ---------------------------------------------------------------------
 */

namespace Glpi\UI;

use Glpi\Application\View\TemplateRenderer;
use InvalidArgumentException;
use RuntimeException;
use Safe\Exceptions\FilesystemException;
use Throwable;

use function Safe\file_get_contents;
use function Safe\json_decode;
use function Safe\mkdir;
use function Safe\realpath;
use function Safe\rename;
use function Safe\unlink;

final class IllustrationManager
{
    private string $icons_definition_file;
    private string $icons_sprites_path;
    private string $scenes_gradient_sprites_path;
    private ?array $icons_definitions = null;

    public const DEFAULT_ILLUSTRATION = "request-service";

    /**
     * Dummy file that is generated by the `php bin/console tools:generate_illustration_translations` command.
     * Its only role is to contains the title of each icons so that the `vendor/bin/extract-locales` script can extract
     * them.
     */
    public const TRANSLATION_FILE = GLPI_ROOT . '/resources/.illustrations_translations.php';

    public const CUSTOM_ILLUSTRATION_PREFIX = "custom:";

    private const CUSTOM_ILLUSTRATION_DIR = GLPI_PICTURE_DIR . "/illustrations";

    public const CUSTOM_SCENE_PREFIX = "custom:";

    private const CUSTOM_SCENES_DIR = GLPI_PICTURE_DIR . "/scenes";

    public function __construct(
        ?string $icons_definition_file = null,
        ?string $icons_sprites_path = null,
        ?string $scenes_gradient_sprites_path = null,
    ) {
        global $CFG_GLPI;

        $this->icons_definition_file = $icons_definition_file ?? GLPI_ROOT
            . '/public/lib/glpi-project/illustrations/icons.json'
        ;
        $this->icons_sprites_path = $icons_sprites_path
            ?? '/lib/glpi-project/illustrations/glpi-illustrations-icons.svg'
        ;
        $this->scenes_gradient_sprites_path = $scenes_gradient_sprites_path
            ?? '/lib/glpi-project/illustrations/glpi-illustrations-scenes-gradient.svg'
        ;

        $this->checkIconFile($this->icons_definition_file);
        $this->checkIconFile(GLPI_ROOT . "/public/$this->scenes_gradient_sprites_path");
        $this->checkIconFile(GLPI_ROOT . "/public/$this->icons_sprites_path");
        $this->validateOrInitCustomContentDir(self::CUSTOM_ILLUSTRATION_DIR);
        $this->validateOrInitCustomContentDir(self::CUSTOM_SCENES_DIR);
    }

    public function getCustomIconIdFromPrefixedString(string $prefixed_id): string
    {
        $custom_icon_prefix = self::CUSTOM_ILLUSTRATION_PREFIX;
        if (!str_starts_with($prefixed_id, $custom_icon_prefix)) {
            throw new InvalidArgumentException("$prefixed_id is not prefixed");
        }
        return substr($prefixed_id, strlen($custom_icon_prefix));
    }

    /**
     * @param int|null $size Height and width (px). Will be set to 100% if null.
     */
    public function renderIcon(string $icon_id, ?int $size = null): string
    {
        $custom_icon_prefix = self::CUSTOM_ILLUSTRATION_PREFIX;
        if (str_starts_with($icon_id, $custom_icon_prefix)) {
            return $this->renderCustomIcon(
                $this->getCustomIconIdFromPrefixedString($icon_id),
                $size
            );
        } else {
            return $this->renderNativeIcon($icon_id, $size);
        }
    }

    /**
     * @param int|null $height Height (px). Will be set to 100% if null.
     * @param int|null $width Width (px). Will be set to 100% if null.
     */
    public function renderScene(
        string $icon_id,
        ?int $height = null,
        ?int $width = null,
    ): string {
        $custom_scene_prefix = self::CUSTOM_SCENE_PREFIX;
        if (str_starts_with($icon_id, $custom_scene_prefix)) {
            return $this->renderCustomScene(
                substr($icon_id, strlen($custom_scene_prefix)),
                $height,
                $width,
            );
        } else {
            return $this->renderNativeScene($icon_id, $height, $width);
        }
    }

    /** @return string[] */
    public function getAllIconsIds(): array
    {
        return array_keys($this->getIconsDefinitions());
    }

    public function countIcons(string $filter = ""): int
    {
        if ($filter == "") {
            return count($this->getIconsDefinitions());
        }

        $icons = array_filter(
            $this->getIconsDefinitions(),
            fn($icon) => str_contains(
                strtolower($icon['title']),
                strtolower($filter),
            )
        );

        return count($icons);
    }

    /** @return string[] */
    public function searchIcons(
        string $filter = "",
        int $page = 1,
        int $page_size = 30,
    ): array {
        $icons = array_filter(
            $this->getIconsDefinitions(),
            fn($icon) => str_contains(
                strtolower(_x("Icon", $icon['title'])),
                strtolower($filter),
            ) || !empty(array_filter(
                $icon['tags'] ?? [],
                fn($tag) => str_contains(
                    strtolower(_x("Icon", $tag)),
                    strtolower($filter)
                ),
            )),
        );

        $icons = array_slice(
            array: $icons,
            offset: ($page - 1) * $page_size,
            length: $page_size,
        );

        return array_keys($icons);
    }

    public function getAllIconsTitles(): array
    {
        $icons = $this->getIconsDefinitions();
        $titles = [];
        foreach ($icons as $icon) {
            $titles[] = $icon['title'];
        }

        return $titles;
    }

    public function getAllIconsTags(): array
    {
        $icons = $this->getIconsDefinitions();
        $tags = [];
        foreach ($icons as $icon) {
            if (isset($icon['tags']) && is_array($icon['tags'])) {
                $tags = array_merge($tags, $icon['tags']);
            }
        }

        return array_unique($tags);
    }

    public function saveCustomIllustration(string $id, string $path): void
    {
        rename($path, self::CUSTOM_ILLUSTRATION_DIR . "/$id");
    }

    public function saveCustomScene(string $id, string $path): void
    {
        rename($path, self::CUSTOM_SCENES_DIR . "/$id");
    }

    public function getCustomIllustrationFile(string $id): ?string
    {
        try {
            $file_path = realpath(self::CUSTOM_ILLUSTRATION_DIR . "/$id");
        } catch (FilesystemException) {
            // File does not exist
            return null;
        }
        $custom_dir_path = realpath(self::CUSTOM_ILLUSTRATION_DIR);

        if (
            // Make sure $id is not maliciously reading from others directories
            !str_starts_with($file_path, $custom_dir_path)
            || !file_exists($file_path)
        ) {
            return null;
        }

        return $file_path;
    }

    public function deleteCustomIllustrationFile(string $id): void
    {
        $file = $this->getCustomIllustrationFile($id);
        if ($file !== null && is_writable(dirname($file))) {
            unlink($file);
        }
    }

    public function getCustomSceneFile(string $id): ?string
    {
        $file_path = realpath(self::CUSTOM_SCENES_DIR . "/$id");
        $custom_dir_path = realpath(self::CUSTOM_SCENES_DIR);

        if (
            // Make sure $id is not maliciously reading from others directories
            !str_starts_with($file_path, $custom_dir_path)
            || !file_exists($file_path)
        ) {
            return null;
        }

        return $file_path;
    }

    private function validateOrInitCustomContentDir(string $dir): void
    {
        if (file_exists($dir)) {
            return;
        }

        try {
            @mkdir($dir);
        } catch (FilesystemException $e) {
            // Race condition: directory may have been created by another concurrent request
            if (!file_exists($dir)) {
                throw $e;
            }
        }
    }

    private function renderNativeIcon(string $icon_id, ?int $size = null): string
    {
        global $TRANSLATE;

        $size = $this->computeSize($size);

        $icons = $this->getIconsDefinitions();

        try {
            // Cannot call `_x()` here as it results in an illegal empty translation `id` when strings are extracted.
            // see #21049
            $title = $TRANSLATE->translate("Icon\004" . ($icons[$icon_id]['title'] ?? ""), 'glpi');
            $title = str_replace("Icon\004", "", $title);
        } catch (Throwable $e) {
            $title = '';
        }

        $twig = TemplateRenderer::getInstance();
        return $twig->render('components/illustration/icon.svg.twig', [
            'file_path' => $this->icons_sprites_path,
            'icon_id'   => $icon_id,
            'width'     => $size,
            'height'    => $size,
            'title'     => $title,
        ]);
    }

    private function renderCustomIcon(string $icon_id, ?int $size = null): string
    {
        $twig = TemplateRenderer::getInstance();
        $size = $this->computeSize($size);
        $url = !empty($icon_id) ? "/UI/Illustration/CustomIllustration/$icon_id" : null;
        return $twig->render('components/illustration/custom_icon.html.twig', [
            'url'    => $url,
            'height' => $size,
            'width'  => $size,
        ]);
    }

    private function renderNativeScene(
        string $icon_id,
        ?int $height = null,
        ?int $width = null,
    ): string {
        $twig = TemplateRenderer::getInstance();
        return $twig->render('components/illustration/icon.svg.twig', [
            'file_path' => $this->scenes_gradient_sprites_path,
            'icon_id'   => $icon_id,
            'height'    => $this->computeSize($height),
            'width'     => $this->computeSize($width),
        ]);
    }

    private function renderCustomScene(
        string $icon_id,
        ?int $height = null,
        ?int $width = null,
    ): string {
        $twig = TemplateRenderer::getInstance();
        $url = !empty($icon_id) ? "/UI/Illustration/CustomScene/$icon_id" : null;
        return $twig->render('components/illustration/custom_icon.html.twig', [
            'url'    => $url,
            'height' => $this->computeSize($height),
            'width'  => $this->computeSize($width),
        ]);
    }

    private function getIconsDefinitions(): array
    {
        if ($this->icons_definitions === null) {
            $json = file_get_contents($this->icons_definition_file);
            $this->icons_definitions = json_decode($json, associative: true, flags: JSON_THROW_ON_ERROR);
        }

        return $this->icons_definitions;
    }

    private function computeSize(?int $size = null): string
    {
        if ($size === null) {
            return "100%";
        }

        return $size . "px";
    }

    private function checkIconFile(string $file): void
    {
        if (!is_readable($file)) {
            throw new RuntimeException("Failed to read file: $file");
        }
    }
}
